/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: pgpDES3Tests.c,v 1.3 1999/12/14 01:04:58 cpeterson Exp $
____________________________________________________________________________*/
/*

DESTEST.C - This file contains functions which are used to verify
the various DES based encryption functions.

TODO - put in good result codes
- check the PGPFree functions for failure?

We want to support the following functions here:

3DES encrypt - ECB
3DES decrypt - ECB
3DES encrypt - CBC
3DES decrypt - CBC
3DES encrypt - CFB
3DES decrypt - CFB

- we can use the same DES key for all of these functions.

*/

/*
FILLDATA will fill up bytes with incrementing values
*/
static void
filldata( unsigned char *data, int size, int start, int incr)
{
unsigned char count=start;

for( ; size--; *data++ = count, count+=incr) {}
}

/*

DESTEST.C - first cut, return a non-zero value if the DES routines
look good.

*/

#include <stdio.h>
#include <string.h>

#include "pgpErrors.h"
#include "pgpMem.h"
#include "pgpUtilities.h"
#include "pgpDES3.h"
#include "pgpSymmetricCipherPriv.h"
#include "pgpCBC.h"
#include "pgpCFB.h"

#define DKEYSIZE 24

/* define the size of the sample data */
#define BSIZE 64

/* test ECB - Electronic Codebook Mode */

	static PGPError
TestDES3ECB(PGPSymmetricCipherContextRef cipherContext)
{
	PGPError 	err = kPGPError_NoErr;
	PGPByte		key[DKEYSIZE];
	PGPByte 	in[BSIZE], out[BSIZE], data[BSIZE];
	
	PGPByte exp[BSIZE] =
	{
		0x26, 0xe0, 0xdd, 0x34, 0xaf, 0x71, 0xfa, 0xba,
		0xb8, 0x04, 0xf0, 0xca, 0x1c, 0x25, 0x30, 0xa6,
		0x8d, 0x67, 0x44, 0xa5, 0xc5, 0x3a, 0xdb, 0x53,
		0xa1, 0x31, 0x8a, 0x4d, 0x39, 0x43, 0x3c, 0xb9,
		0x42, 0x2c, 0x33, 0x82, 0xc5, 0xf6, 0xf1, 0xa3,
		0x90, 0xb1, 0x05, 0xc0, 0x4f, 0x2e, 0x10, 0x0e,
	 	0x70, 0xc0, 0x39, 0x9a, 0x74, 0xf2, 0x54, 0x1c,
		0x0f, 0x69, 0xe5, 0xc1, 0x1a, 0x56, 0x0a, 0x4f
	};

	/* set the key */
	filldata( key, sizeof( key ), 0, 3);
	filldata( data, sizeof( data ), 0, 1);

	err = PGPInitSymmetricCipher( cipherContext, key );
	if( IsntPGPError( err ) )
	{
		PGPUInt32	i;
		
		/* do an encryption, check the result */
		for( i = 0; i < sizeof( data ); i += 8 )
		{
			err = PGPSymmetricCipherEncrypt( cipherContext,
						data + i, out + i );
			if( IsPGPError( err ) )
				break;
		}
	
		if( IsntPGPError( err ) &&
			! pgpMemoryEqual( out, exp, sizeof( exp ) ) )
		{
			err = kPGPError_SelfTestFailed;
		}
	}

	if( IsntPGPError( err ) )
	{
		err = PGPInitSymmetricCipher( cipherContext, key );
		if( IsntPGPError( err ) )
		{
			PGPUInt32	i;

			/* do an decryption, check the result */
			for( i = 0; i < sizeof( data ); i += 8 )
			{
				err = PGPSymmetricCipherDecrypt( cipherContext,
							out + i, in + i );
				if( IsPGPError( err ) )
					break;
			}
		
			if( IsntPGPError( err ) &&
				! pgpMemoryEqual( in, data, sizeof( data ) ) )
			{
				err = kPGPError_SelfTestFailed;
			}
		}
	}
	
	return( err );
}

/* test CBC - Cipher Block Chaining */

	static PGPError
TestDES3CBC(PGPSymmetricCipherContextRef cipherContext)
{
	PGPCBCContextRef 				cbcContext;
	PGPByte 						key[DKEYSIZE];
	PGPByte 						iv[8];
	PGPByte 						in[BSIZE], out[BSIZE], data[BSIZE];
	PGPError						err;
	PGPSymmetricCipherContextRef 	cipherContextCopy;
	
	PGPByte exp[BSIZE] =
	{
		0x36, 0x97, 0xc1, 0x48, 0xbe, 0x18, 0x24, 0x7c,
		0xa6, 0xab, 0x7c, 0x43, 0x62, 0xbe, 0x7f, 0x6b,
		0xb4, 0xa3, 0xa0, 0x53, 0x89, 0x38, 0xe2, 0xc5,
		0x18, 0xa2, 0xcd, 0xef, 0x9c, 0x91, 0x9b, 0xd1,
		0x16, 0x72, 0x9f, 0xcf, 0x0a, 0x0f, 0x4d, 0x10,
		0x1e, 0xcd, 0x1c, 0x8a, 0x0e, 0x47, 0x49, 0x45,
		0x88, 0x2c, 0x0c, 0xe9, 0x3b, 0x0b, 0x9b, 0xc9,
		0xd0, 0xbe, 0x0f, 0xd5, 0x9d, 0xdc, 0x2d, 0xf5
	};

	/* create a 3DES CBC */
	/* and test it */
	filldata( key, sizeof( key ), 0, 3 );
	filldata( iv, sizeof( iv ), 1, 5 );
	filldata( data, sizeof( data ), 0, 1 );

	err = PGPCopySymmetricCipherContext( cipherContext, &cipherContextCopy );
	if( IsntPGPError( err ) )
	{	
		err = PGPNewCBCContext( cipherContextCopy, &cbcContext );
		if( IsntPGPError( err ) )
		{
			err = PGPInitCBC( cbcContext, key, iv );
			if( IsntPGPError( err ) )
			{
				PGPUInt32	i;
				
				for ( i = 0; i < sizeof( data ); i += 8 )
				{
					err = PGPCBCEncrypt( cbcContext, data + i, 8, out + i );
					if( IsPGPError( err ) )
						break;
				}

				if( IsntPGPError( err ) &&
					! pgpMemoryEqual( out, exp, sizeof( exp ) ) )
				{
					err = kPGPError_SelfTestFailed;
				}
			}
			
			if( IsntPGPError( err ) )
			{
				err = PGPInitCBC( cbcContext, key, iv );
				if( IsntPGPError( err ) )
				{
					PGPUInt32	i;
					
					for ( i = 0; i < sizeof( data ); i += 8 )
					{
						err = PGPCBCDecrypt( cbcContext, out + i, 8, in + i );
						if( IsPGPError( err ) )
							break;
					}

					if( IsntPGPError( err ) &&
						! pgpMemoryEqual( in, data, sizeof( data ) ) )
					{
						err = kPGPError_SelfTestFailed;
					}
				}
			}
			
			(void) PGPFreeCBCContext( cbcContext );
		}
	}
	
	return( err );
}

/* test CFB - Cipher Feedback Mode */

	static PGPError
TestDES3CFB(PGPSymmetricCipherContextRef cipherContext)
{
	PGPCFBContextRef 				cfbContext;
	PGPByte 						key[DKEYSIZE];
	PGPByte 						iv[8];
	PGPByte 						in[BSIZE], out[BSIZE], data[BSIZE];
	PGPError						err;
	PGPSymmetricCipherContextRef 	cipherContextCopy;
	
	PGPByte exp[BSIZE] =
	{
		0xe0, 0x83, 0x1d, 0xa5, 0x70, 0xd0, 0x5d, 0xab,
		0x28, 0x3e, 0x39, 0x8c, 0x1a, 0x74, 0x9a, 0xd1,
		0xb6, 0x25, 0xb7, 0x85, 0xf3, 0x82, 0xe3, 0x8d,
		0xb0, 0xc2, 0x65, 0xd6, 0x9c, 0x30, 0x9c, 0xf1,
		0x1c, 0x66, 0x0b, 0xd8, 0x9d, 0xdf, 0x64, 0xbb,
		0x3e, 0x74, 0x48, 0xa4, 0x7b, 0xca, 0x46, 0xec,
		0xfe, 0xf4, 0xa4, 0x26, 0xb8, 0x5f, 0x51, 0xcd,
		0x0d, 0xac, 0xcd, 0xac, 0xb7, 0xb7, 0xf7, 0x36
	};

	/* create a 3DES CFB and test it */
	filldata( key, sizeof( key ), 0, 3 );
	filldata( iv, sizeof( iv ), 1, 5 );
	filldata( data, sizeof( data ), 0, 1 );

	err = PGPCopySymmetricCipherContext( cipherContext, &cipherContextCopy );
	if( IsntPGPError( err ) )
	{	
		err = PGPNewCFBContext( cipherContextCopy, 1, &cfbContext );
		if( IsntPGPError( err ) )
		{
			err = PGPInitCFB( cfbContext, key, iv );
			if( IsntPGPError( err ) )
			{
				PGPUInt32	i;
				
				for ( i = 0; i < sizeof( data ); i += 8 )
				{
					err = PGPCFBEncrypt( cfbContext, data + i, 8, out + i );
					if( IsPGPError( err ) )
						break;
				}

				if( IsntPGPError( err ) &&
					! pgpMemoryEqual( out, exp, sizeof( exp ) ) )
				{
					err = kPGPError_SelfTestFailed;
				}
			}
			
			if( IsntPGPError( err ) )
			{
				err = PGPInitCFB( cfbContext, key, iv );
				if( IsntPGPError( err ) )
				{
					PGPUInt32	i;
					
					for ( i = 0; i < sizeof( data ); i += 8 )
					{
						err = PGPCFBDecrypt( cfbContext, out + i, 8, in + i );
						if( IsPGPError( err ) )
							break;
					}

					if( IsntPGPError( err ) &&
						! pgpMemoryEqual( in, data, sizeof( data ) ) )
					{
						err = kPGPError_SelfTestFailed;
					}
				}
			}
			
			(void) PGPFreeCFBContext( cfbContext );
		}
	}
	
	return( err );
}

/* FIPS140 state Run DES Test */
	PGPError
pgpTest3DES(void)
{
	PGPError		err;
	PGPContextRef	context;
	PGPCustomContextInfo	info;
	
	pgpClearMemory( &info, sizeof( info ) );
	
	info.sdkAPIVersion 	= kPGPsdkAPIVersion;
	info.memoryMgr		= PGPGetDefaultMemoryMgr();
	
	err = PGPNewContextCustom( &info, &context );
	if( IsntPGPError( err ) )
	{
		PGPSymmetricCipherContextRef 	cipherContext;
			
		err = PGPNewSymmetricCipherContext( context, kPGPCipherAlgorithm_3DES,
					 &cipherContext );
		if( IsntPGPError( err ) )
		{
			err = TestDES3ECB( cipherContext );
			if( IsntPGPError( err ) )
			{
				err = TestDES3CBC( cipherContext );
				if( IsntPGPError( err ) )
				{
					err = TestDES3CFB( cipherContext );
				}
			}
			
			(void) PGPFreeSymmetricCipherContext( cipherContext );
		}
		
		PGPFreeContext( context );
	}
	
	/*
	** For the purposes of a self-test, only kPGPError_SelfTestFailed is
	** ever returned as an error code.
	*/
	
	if( IsPGPError( err ) )
		err = kPGPError_SelfTestFailed;
		
	return( err );
}
